﻿using log4net;
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using System.Xml.Schema;
using VA.PPMS.IWS.Common;
using VA.PPMS.IWS.Functions.Configuration.Interface;
using VA.PPMS.IWS.SchemaValidationService.Interface;
using VA.PPMS.ProviderData;
using static VA.PPMS.IWS.Functions.Configuration.Interface.SchemaOptions;

namespace VA.PPMS.IWS.SchemaValidationService
{
    public class SchemaValidationService : ISchemaValidationService
    {
        private readonly ILog _logger;
        private readonly IIwsConfiguration _configuration;

        public SchemaValidationService(ILog logger, IIwsConfiguration configuration)
        {
            _logger = logger;
            _configuration = configuration;
        }

        public async Task<List<Validation>> ValidateSchemaAsync(Providers providers)
        {
            try
            {
                _logger.Info($"@@@@ INFO - Start ValidateSchemaAsync for TransactionId: {providers.TransactionId} @@@@");

                var schemaOption = providers.IsVaNetwork ? SchemaProfiles.VA : SchemaProfiles.CCN;
                var packet = await _configuration.GetSchemaProfileAsync(schemaOption);
                
                var returnValue = new List<Validation>();
                var providerCollectionResult = new ConcurrentDictionary<string, Validation>();

                var suffix = 1;

                foreach (var provider in providers.Provider)
                {
#if DEBUG
                    _logger.Debug($"---- DEBUG - Schema Validation for ProviderId: {provider.ProviderId} ----" );
#endif
                    // If insert, validate against schema
                    var validationResult = provider.IsModification ? string.Empty : ValidateSchema(provider, packet);
                    var id = provider.ProviderId;
                    
                    // Check to make sure ProviderId is unique in batch
                    if (providerCollectionResult.ContainsKey(id))
                    {
                        id = $"{id}-{suffix++}";
                        validationResult = "Duplicate ProviderId";
                    }

                    providerCollectionResult.TryAdd(id, new Validation(provider.ProviderId, provider.CorrelationId, provider.ProviderNameDisplay, (int)provider.TransactionType, validationResult));
                }

                foreach (var validation in providerCollectionResult)
                {
                    returnValue.Add(validation.Value);
                }

                providerCollectionResult.Clear();

                _logger.Info($"@@@@ INFO - End ValidateSchemaAsync for TransactionId: {providers.TransactionId} @@@@");

                return returnValue;
            }
            catch (Exception ex)
            {
                _logger.Error($"@@@@ ERROR - There was a problem with the Schema Validation Service for TransactionId: {providers.TransactionId} @@@@", ex);
                throw new PpmsServiceException($"There was a problem with the Schema Validation Service for TransactionId: {providers.TransactionId}", ex);
            }
        }

        private static string ValidateSchema<T>(T classInstance, SchemaProfile profile)
        {
            var payload = Utilities.SerializeInstance(classInstance, profile.Prefix, profile.Namespace);

            var sb = new StringBuilder();
            var customDataFilePath = $"{profile.SchemaName}";

            var xmlReaderSettings = new XmlReaderSettings();
            xmlReaderSettings.Schemas.Add(profile.Namespace, customDataFilePath);
            xmlReaderSettings.ValidationType = ValidationType.Schema;
            xmlReaderSettings.ValidationEventHandler += (o, e) => { if (e.Severity == XmlSeverityType.Error) sb.Append(e.Message); };

            var reader = XmlReader.Create(new StringReader(payload), xmlReaderSettings);

            while (reader.Read()) {}

            return sb.ToString();
        }
    }
}